/*
 Program WinCaml: Graphical User Interface
 for interactive use of Caml-Light and Ocaml.
 Copyright (C) 2005-2018 Jean Mouric 35700 Rennes France
 email: jean.mouric@orange.fr
 
 This program is free software: you can redistribute it and/or modify
 it under the terms of the GNU General Public License as published by
 the Free Software Foundation, either version 3 of the License, or
 (at your option) any later version.
 
 This program is distributed in the hope that it will be useful,
 but WITHOUT ANY WARRANTY; without even the implied warranty of
 MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 GNU General Public License for more details.
 
 You should have received a copy of the GNU General Public License
 along with this program. If not, see <http://www.gnu.org/licenses/>.
 */

// File wx/Paths.cpp

#include "wx/wx.h"
#include "wxw.h"
#include "wx/file.h"
#include "wx/stdpaths.h"
#include "wx/filename.h"

extern bool camlnew;

using namespace std;

#ifndef __WXMSW__
#include <stdio.h>
#include <sys/stat.h>

string toBytes(const wstring&);
wxString applicationDirPath();
bool createDirectory(const wstring& path);
bool getConfigFolderPath(wstring& path);
CProcess* startCaml(bool ocaml);
void camllightHelp();
void ocamlHelp();
wstring setOCamlPath();
wstring getOCamlPath();
void ldconf(const wstring& ocamlPath);
bool resetCamlPaths();

wxString applicationDirPath()
{
    wxStandardPaths paths = wxStandardPaths::Get();
#ifdef __WXMAC__
    return paths.GetExecutablePath().BeforeLast('/').BeforeLast('/').BeforeLast('/').BeforeLast('/');
#else
    return paths.GetExecutablePath().BeforeLast('/');
#endif
}

bool getConfigFolderPath(wstring& path)
{
#ifdef __WXMAC__
    wxString s = wxStandardPaths::Get().GetUserLocalDataDir();
    if (s != "") {
        path = s.ToStdWstring() + L"/";
        return true;
    }
    return false;
#else
    wxString s = wxGetUserId();
    if (s != "") {
	s = applicationDirPath() + "/." + s;
        path = s.ToStdWstring() + L"/"; 
	return true;
    }
    return false;
#endif
}

void ldconf(const wstring& ocamlPath)
{
	wstring ldconfPath = ocamlPath +  L"/lib/ocaml/ld.conf";
	string toBytes(const wstring& wstr);
    fstream fs(toBytes(ldconfPath).c_str(), fstream::out);
    if (!fs.fail())
    {
        fs << wxString(ocamlPath + L"/lib/ocaml/stublibs").utf8_str() << endl;
        fs << wxString(ocamlPath + L"/lib/ocaml").utf8_str();
        fs.close();
    }
}

wstring getCamlPath(const wstring& camlbase)
{
    wstring cfp;
    string camlPath;
    if (getConfigFolderPath(cfp))
    {
        cfp += camlbase;
        string toBytes(const wstring& wstr);
        fstream fs(toBytes(cfp).c_str(), fstream::in);
        if (!fs.fail())
        {
            char c;
            while (fs.get(c) && c != '\r' && c != '\n')
                camlPath += c;
            
            fs.close();
        }
   }
   return (wxString::FromUTF8(camlPath.c_str())).ToStdWstring();
}

void writeCamlPath(wxString camlbase, wxString camlPath)
{
    wstring s;
    if (getConfigFolderPath(s) && createDirectory(s))
    {
        s += camlbase.ToStdWstring();
        string toBytes(const wstring& wstr);
        fstream fs(toBytes(s).c_str(), fstream::out);
        if (!fs.fail())
        {
            fs << string(camlPath.utf8_str()) << endl;
            fs.close();
        }
    }
}

wxString setCamlPath(bool ocaml, bool& failure)
{
    wstring distrib = ocaml ? L"OCaml" : L"Caml Light";
    wstring message = L"Le dialogue suivant devrait permettre de naviguer vers un dossier " + distrib + L" valide ou d'annuler.";
    ::infoMessage(message);
    wxString prompt = ocaml ? "Emplacement d'OCaml ?" : "Emplacement de Caml Light ?";
    wxDirDialog dlg(NULL, prompt, wxEmptyString, wxDD_DIR_MUST_EXIST);
    failure = dlg.ShowModal() != wxID_OK;
    return failure ? "" : dlg.GetPath();
}

bool resetCamlPath(const wxString& camlbase)
{
    wstring s;
    if (getConfigFolderPath(s))
    {
        s += camlbase.ToStdWstring();
        if (wxFile::Exists(wxString(s)))
        {
			wxString distrib = camlbase == "camlbase.txt" ? "Caml Light" : "OCaml";
			wxString message = "Abandonner la distribution " + distrib + " actuelle ?";
			if (::yesnoMessage(message.ToStdWstring()))
			{
				void deleteFile(const wstring& fileName);
				deleteFile(s);
			}
			return true;
        }
    }
	return false;
}

bool resetCamlPaths()
{
    bool b1 = resetCamlPath("camlbase.txt");
    bool b2 = resetCamlPath("ocamlbase.txt");
	return b1 || b2;
}

#ifndef __WXMAC__
char byte64(const char* path)
{
    unsigned char c[5] = {0};
    FILE* f = fopen(path, "rb");
    if (!f) return 0;
    if (fread(c, 5, 1, f) == 1)
    {
        fclose(f);
        return c[4];
    }
    return 0;
}
#endif

bool isExecutable(const wxString& runtimePath)
{
    FILE* f = fopen(runtimePath.ToStdString().c_str(), "rb");
    if (!f) return false;
    char c[5];
    if (fread(c, 5, 1, f) == 1)
    {
        fclose(f);
#ifdef __WXMAC__
        uint32_t m = *(uint32_t*)c;
        if (m != 0xfeedface && m != 0xfeedfacf && m!= 0xcefaedfe && m!= 0xcffaedfe)
#else
        if (c[0] != 0x7f || c[1] != 0x45 || c[2] != 0x4c || c[3] != 0x46 || c[4] != byte64("/bin/sh"))
#endif
        {
            errorMessage(L"Format de fichier binaire incorrect");
            return false;
        }
    }
    return true;
}

#ifdef __WXMAC__
bool isMac64Executable(const char* path)
{
    unsigned char c[4] = {0};
    FILE* f = fopen(path, "rb");
    if (!f) return false;
    if (fread(c, 4, 1, f) == 1)
    {
        fclose(f);
        return c[0] == 0xcf && c[1] == 0xfa && c[2] == 0xed && c[3] == 0xfe;
    }
    return false;
}
#else
bool isElf64leExecutable(const char* path)
{
    unsigned char c[6] = {0};
    FILE* f = fopen(path, "rb");
    if (!f) return false;
    if (fread(c, 6, 1, f) == 1)
    {
        fclose(f);
        return c[0] == 0x7f && c[1] == 0x45 && c[2] == 0x4c && c[3] == 0x46 && c[4] == 0x02 && c[5] == 0x01;
    }
    return false;
}
#endif

bool findOCaml(wxString& camlPath)
{
    wxFileName fi1 = wxString("/usr/local/bin/ocamlrun");
    wxFileName fi2 = wxString("/usr/local/bin/ocaml");
    if (!fi1.FileExists() || !fi2.FileExists() || !fi1.IsFileExecutable() || !isExecutable("/usr/local/bin/ocamlrun"))
    {
        camlPath = "/usr";
        fi1 = wxString("/usr/bin/ocamlrun");
        fi2 = wxString("/usr/bin/ocaml");
        if (!fi1.FileExists() || !fi2.FileExists() || !fi1.IsFileExecutable() || !isExecutable("/usr/bin/ocamlrun"))
        {
            camlPath = wxString(getCamlPath(L"ocamlbase.txt"));
            fi1 = camlPath + "/bin/ocamlrun";
            fi2 = camlPath + "/bin/ocaml";
            while (!fi1.FileExists() || !fi2.FileExists() || !fi1.IsFileExecutable() || !isExecutable(camlPath + "/bin/ocamlrun"))
            {
                bool failure;
                wxString path = setCamlPath(true, failure);
                if (failure) return false;
                camlPath = path;
                fi1 = camlPath + "/bin/ocamlrun";
                fi2 = camlPath + "/bin/ocaml";
            }
        }
    }
    return true;
}

bool findCamllight(wxString& camlPath, wxString& toplevel, wxString& camlLib)
{
    wxFileName fi = toplevel;
    if (!fi.FileExists() || !fi.IsFileExecutable() || !isExecutable(toplevel))
    {
        camlPath = "/usr/local";
        camlLib = camlPath + "/lib/caml-light";
        toplevel = camlLib + "/caml_all";
        fi = toplevel;
        if (!fi.FileExists() || !fi.IsFileExecutable() || !isExecutable(toplevel))
        {
            camlPath = "/usr";
            camlLib = camlPath + "/lib/caml-light";
            toplevel = camlLib + "/caml_all";
            fi = toplevel;
            if (!fi.FileExists() || !fi.IsFileExecutable() || !isExecutable(toplevel))
            {
                camlPath = wxString(getCamlPath(L"camlbase.txt"));
                camlLib = camlPath + "/lib/caml-light";
                toplevel = camlLib + "/caml_all";
                fi = toplevel;
                while (!fi.FileExists() || !fi.IsFileExecutable() || !isExecutable(toplevel))
                {
                    bool failure;
                    wxString path = setCamlPath(false, failure);
                    if (failure) return false;
                    camlPath = path;
                    camlLib = camlPath + "/lib/caml-light";
                    toplevel = camlLib + "/caml_all";
                    fi = toplevel;
                }
            }
        }
        
    }
    return true;
}

bool findOCaml1(wxString& camlPath)
{
    bool failure;
    wxString path = setCamlPath(true, failure);
    if (failure) {camlPath = ""; return false;}
    wxFileName fi1 = path + "/bin/ocamlrun";
    wxFileName fi2 = path + "/bin/ocaml";
    while (!fi1.FileExists() || !fi2.FileExists() || !fi1.IsFileExecutable() || !isExecutable(path + "/bin/ocamlrun"))
    {
        path = setCamlPath(true, failure);
        if (failure) {camlPath = ""; return false;}
        //camlPath = path;
        fi1 = camlPath + "/bin/ocamlrun";
        fi2 = camlPath + "/bin/ocaml";
    }
    camlPath = path;
    return true;
}

bool findCamllight1(wxString& camlPath, wxString& toplevel, wxString& camlLib)
{
    bool failure;
    camlPath = setCamlPath(false, failure);
    if (failure) {camlPath = ""; return false;}
    camlLib = camlPath + "/lib/caml-light";
    toplevel = camlLib + "/caml_all";
    wxFileName fi = toplevel;
    while (!fi.FileExists() || !fi.IsFileExecutable() || !isExecutable(toplevel))
    {
        camlPath = setCamlPath(false, failure);
        if (failure)  {camlPath = ""; return false;}
        camlLib = camlPath + "/lib/caml-light";
        toplevel = camlLib + "/caml_all";
        fi = toplevel;
    }
    return true;
}

CProcess* startCaml1(bool ocaml)
{
    wxSetWorkingDirectory(applicationDirPath());
   if (ocaml)
    {
        wxString ocamlPath = "/usr/local";
        if (!findOCaml(ocamlPath)) return NULL;
        if (ocamlPath != "/usr/local" && ocamlPath != "/usr") writeCamlPath("ocamlbase.txt", ocamlPath);
        
#ifdef __WXMAC__
        wxString graphlib = isMac64Executable((ocamlPath + "/bin/ocamlrun").ToStdString().c_str()) ? "\"" + applicationDirPath() + "/ocaml/mac/ocaml64\"" : "\"" + applicationDirPath() + "/ocaml/mac/ocaml32\"";
#else
        wxString graphlib =  isElf64leExecutable((ocamlPath + "/bin/ocamlrun").ToStdString().c_str()) ? "\"" + applicationDirPath() + "/ocaml/linux/ocaml64\"" : "\"" + applicationDirPath() + "/ocaml/linux/ocaml32\"";
#endif

        ldconf(ocamlPath.ToStdWstring());

        wxSetEnv("OCAMLLIB", ocamlPath + "/lib/ocaml");
        wxSetEnv("CAML_LD_LIBRARY_PATH", ocamlPath + "/lib/ocaml/stublibs");
        wxSetEnv("PATH", ocamlPath + "/bin:/usr/local/bin:/usr/bin:/bin");

        wxString prefix = "\"" + ocamlPath;
        wxString cmdline = prefix + "/bin/ocamlrun\" " + prefix + "/bin/ocaml\"";
        cmdline += " -I " + graphlib;
        cmdline += " -I " + prefix + "/lib/ocaml\"";
        cmdline += " -I " + prefix + "/lib/ocaml/stublibs\"";

        CProcess* process = new CProcess();
        process->start(cmdline.ToStdString());
        return process;
    }
    else
    {
        wxString camlPath = applicationDirPath() + "/caml-light";
        wxString path = camlPath;
        wxString camlLib = camlPath + "/lib/caml-light";

#ifdef __WXMAC__
        wxString toplevel = isMac64Executable("/bin/sh") ? camlPath + "/bin/mactoplevel64" : camlPath+ "/bin/mactoplevel32";
#else
        wxString toplevel = isElf64leExecutable("/bin/sh") ? camlPath + "/bin/lintoplevel64" : camlPath + "/bin/lintoplevel32";
#endif
        if (!findCamllight(camlPath, toplevel, camlLib)) return NULL;

        if (camlPath != "/usr/local" && camlPath != "/usr" && camlPath != path)
        {
            writeCamlPath("camlbase.txt", camlPath);
        }

        wxSetEnv("PATH", camlPath + "/bin:/usr/local/bin:/usr/bin:/bin");
        wxSetEnv("CAMLLIGHTDIR", camlPath);
        
        wxLocale locale(wxLocale::GetSystemLanguage());
        wxString commandLine = "\"" + toplevel + "\" -stdlib \"" + camlLib + "\" -lang " + locale.GetCanonicalName().SubString(0, 1);
        CProcess* process = new CProcess();
        process->start(commandLine.ToStdString());
        return process;
    
    }
}

CProcess* startCaml(bool ocaml)
{
    wstring message = L"Changer de distribution ";
    message += (ocaml ? L"OCaml ?" : L"Caml Light ?");
    wxSetWorkingDirectory(applicationDirPath());
    if (camlnew && ::yesnoMessage(message))
    {
        camlnew = false;
        if (ocaml)
        {
            wxString camlPath;
            if (findOCaml1(camlPath))
            {
#ifdef __WXMAC__
                wxString graphlib = isMac64Executable((camlPath + "/bin/ocamlrun").ToStdString().c_str()) ? "\"" + applicationDirPath() + "/ocaml/mac/ocaml64\"" : "\"" + applicationDirPath() + "/ocaml/mac/ocaml32\"";
#else
                wxString graphlib =  isElf64leExecutable((camlPath + "/bin/ocamlrun").ToStdString().c_str()) ? "\"" + applicationDirPath() + "/ocaml/linux/ocaml64\"" : "\"" + applicationDirPath() + "/ocaml/linux/ocaml32\"";
#endif
                writeCamlPath("ocamlbase.txt", camlPath);
                ldconf(camlPath.ToStdWstring());
                
                wxSetEnv("OCAMLLIB", camlPath + "/lib/ocaml");
                wxSetEnv("CAML_LD_LIBRARY_PATH", camlPath + "/lib/ocaml/stublibs");
                wxSetEnv("PATH", camlPath + "/bin:/usr/local/bin:/usr/bin:/bin");
                
                wxString prefix = "\"" + camlPath;
                wxString cmdline = prefix + "/bin/ocamlrun\" " + prefix + "/bin/ocaml\"";
                cmdline += " -I " + graphlib;
                cmdline += " -I " + prefix + "/lib/ocaml\"";
                cmdline += " -I " + prefix + "/lib/ocaml/stublibs\"";
                
                CProcess* process = new CProcess();
                process->start(cmdline.ToStdString());
                return process;
            }
            else
            {
                if (getCamlPath(L"ocamlbase.txt") != L"" && ::yesnoMessage(L"abandonner la distribution OCaml actuelle ?"))
                {
                    resetCamlPath("ocamlbase.txt");
                    return startCaml1(ocaml);
                }
            }
        }
        else
        {
            wxString camlPath;
            wxString toplevel;
            wxString camlLib;
            if (findCamllight1(camlPath, toplevel, camlLib))
            {
                writeCamlPath("camlbase.txt", camlPath);
                
                wxSetEnv("PATH", camlPath + "/bin:/usr/local/bin:/usr/bin:/bin");
                wxSetEnv("CAMLLIGHTDIR", camlPath);
                
                wxLocale locale(wxLocale::GetSystemLanguage());
                wxString commandLine = "\"" + toplevel + "\" -stdlib \"" + camlLib + "\" -lang " + locale.GetCanonicalName().SubString(0, 1);
                
                CProcess* process = new CProcess();
                process->start(commandLine.ToStdString());
                return process;
            }
            else
            {
                if (getCamlPath(L"camlbase.txt") != L"" && ::yesnoMessage(L"abandonner la distribution Caml Light actuelle ?"))
                {
                    resetCamlPath("camlbase.txt");
                    return startCaml1(ocaml);
                }
            }
        }
    }
    camlnew = false;
    if (ocaml)
    {
        wxString camlPath = wxString(getCamlPath(L"ocamlbase.txt"));
        wxFileName fi1 = camlPath + "/bin/ocamlrun";
        wxFileName fi2 = camlPath + "/bin/ocaml";
        if (!fi1.FileExists() || !fi2.FileExists() || !fi1.IsFileExecutable() || !isExecutable(camlPath + "/bin/ocamlrun"))
        {
            return startCaml1(ocaml);
        }
        else
        {
#ifdef __WXMAC__
            wxString graphlib = isMac64Executable((camlPath + "/bin/ocamlrun").ToStdString().c_str()) ? "\"" + applicationDirPath() + "/ocaml/mac/ocaml64\"" : "\"" + applicationDirPath() + "/ocaml/mac/ocaml32\"";
#else
            wxString graphlib =  isElf64leExecutable((camlPath + "/bin/ocamlrun").ToStdString().c_str()) ? "\"" + applicationDirPath() + "/ocaml/linux/ocaml64\"" : "\"" + applicationDirPath() + "/ocaml/linux/ocaml32\"";
#endif
            wxSetEnv("OCAMLLIB", camlPath + "/lib/ocaml");
            wxSetEnv("CAML_LD_LIBRARY_PATH", camlPath + "/lib/ocaml/stublibs");
            wxSetEnv("PATH", camlPath + "/bin:/usr/local/bin:/usr/bin:/bin");
            
            wxString prefix = "\"" + camlPath;
            wxString cmdline = prefix + "/bin/ocamlrun\" " + prefix + "/bin/ocaml\"";
            cmdline += " -I " + graphlib;
            cmdline += " -I " + prefix + "/lib/ocaml\"";
            cmdline += " -I " + prefix + "/lib/ocaml/stublibs\"";
            
            CProcess* process = new CProcess();
            process->start(cmdline.ToStdString());
            return process;

        }
    }
    else
    {
        wxString camlPath = wxString(getCamlPath(L"camlbase.txt"));
        wxString camlLib = camlPath + "/lib/caml-light";
        wxString toplevel = camlLib + "/caml_all";
        wxFileName fi = toplevel;
        if (!fi.FileExists() || !fi.IsFileExecutable() || !isExecutable(toplevel))
        {
            return startCaml1(ocaml);
        }
        else
        {
            wxSetEnv("PATH", camlPath + "/bin:/usr/local/bin:/usr/bin:/bin");
            wxSetEnv("CAMLLIGHTDIR", camlPath);
            
            wxLocale locale(wxLocale::GetSystemLanguage());
            wxString commandLine = "\"" + toplevel + "\" -stdlib \"" + camlLib + "\" -lang " + locale.GetCanonicalName().SubString(0, 1);
            
            CProcess* process = new CProcess();
            process->start(commandLine.ToStdString());
            return process;
        }
    }
}

void camllightHelp()
{
#ifdef __WXMAC__
    wxExecute("open -a safari.app \"" + applicationDirPath() + "/doc/man-caml/index.html\"");
#else
    wxExecute("/usr/bin/firefox \"" + applicationDirPath() + "/doc/man-caml/index.html\"");
#endif
}

void ocamlHelp()
{
#ifdef __WXMAC__
    wxExecute("open -a safari.app \"" + applicationDirPath() + "/doc/htmlman/index.html\"");
#else
    wxExecute("/usr/bin/firefox \"" + applicationDirPath() + "/doc/htmlman/index.html\"");
#endif
}
#endif

#ifdef __WXMSW__ 
#include "../Win/Paths.cpp"
#endif
