/*
Program WinCaml: Graphical User Interface
for interactive use of Caml-Light and Ocaml.
Copyright (C) 2005-2017 Jean Mouric 35700 Rennes France
email: jean.mouric@orange.fr

This program is free software: you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation, either version 3 of the License, or
(at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program. If not, see <http://www.gnu.org/licenses/>.
*/

// file Win/CProcess.h   (included from Win/Win.h, Qt/Qt.h and wx/wxw.h)

#include <stdio.h>
#include <fstream>
#define BUFFERSIZE 32768
wstring applicationDirectoryPath();

class CProcess
{
public:
    CProcess()
    {
        ::memset(&pi, 0, sizeof(PROCESS_INFORMATION));
        ::memset(&startInfo, 0, sizeof(STARTUPINFO));
        exitCode = 0;
        buffer = new char[BUFFERSIZE];
        ::memset(bashpid, 0, 100);
		appDirPath = applicationDirectoryPath();
    }
    ~CProcess()
    {
        writeToPipe("\015\012");
        Sleep(100);
        delete[] buffer;		
    }
    void setPid(string cygwin) // Cygwin
    {
        if (cygwin != "")
        {
			sprintf(bashpid, "%s\\%d", cygwin.c_str(), (int)pi.dwProcessId);			
            FILE* f = fopen(bashpid, "w");
            if (f)
			{
				fclose(f);
			}
        }
    }
    string read()
    {
        int count;
        while ((count = readFromPipe(buffer, BUFFERSIZE - 1)) == 0);
        buffer[count] = 0;
        string st(buffer);
        return st;
    }
    bool isRunning()
    {
        if (pi.hProcess) ::GetExitCodeProcess(pi.hProcess, &exitCode);
        else exitCode = 0;
        return exitCode == STILL_ACTIVE;
    }
    bool interrupt()
    {
        if (!isRunning()) return true;
        if (bashpid[0] != 0)//cygwin
        {
            FILE *f;
            f = fopen(bashpid, "w");
            if (f)
            {
                fprintf(f, "%c", 'i');
                fclose(f);
                Sleep(100);
                writeToPipe("\015\012");
                return true;
            }
        }
        else if (::GenerateConsoleCtrlEvent(CTRL_BREAK_EVENT, pi.dwProcessId))
        {
            Sleep(100);
            writeToPipe("\015\012");
            return true;
        }
        return false;
    }
    string readAllStandardOutput()
    {
        bA.erase();
        for (;;)
        {
            DWORD count = readFromPipe(buffer, BUFFERSIZE - 1);
            if (count == 0) break;
            buffer[count] = '\0';
            bA.append(buffer);
        }
        return bA;
    }
    bool start(const wstring& commandLine)
    {
        ::memset(&pi, 0, sizeof(PROCESS_INFORMATION));
        ::memset(&startInfo, 0, sizeof(STARTUPINFO));
        exitCode = 0;
        
        SECURITY_ATTRIBUTES saAttr;
        saAttr.nLength = sizeof(SECURITY_ATTRIBUTES);
        saAttr.bInheritHandle = TRUE;
        saAttr.lpSecurityDescriptor = NULL;
        
        if (!::CreatePipe(&hStdoutRd, &hStdoutWr, &saAttr, 8192))
            return false;
        
        if (!::CreatePipe(&hStdinRd, &hStdinWr, &saAttr, 8192))
            return false;
        
        startInfo.dwFlags = STARTF_USESTDHANDLES | STARTF_USESHOWWINDOW | CREATE_UNICODE_ENVIRONMENT;
        startInfo.wShowWindow = SW_SHOW;
        startInfo.hStdOutput = hStdoutWr;
        startInfo.hStdError = hStdoutWr;
        startInfo.hStdInput = hStdinRd;
        
		appDirPath = applicationDirectoryPath();
        
		return ::CreateProcess(NULL, LPWSTR(commandLine.c_str()), NULL, NULL, TRUE,
                               CREATE_UNICODE_ENVIRONMENT | CREATE_NEW_PROCESS_GROUP | NORMAL_PRIORITY_CLASS, (LPWSTR)GetEnvironmentStrings(), LPWSTR(appDirPath.c_str()), &startInfo, &pi
                               ) ? true : false;
    }
    void stop()
    {
        if (bashpid[0] != 0)//cygwin
        {
            FILE *f;
            f = fopen(bashpid, "w");
            if (f)
            {
                fprintf(f, "%c", 't');
                fclose(f);
            }
            Sleep(100);
            f = fopen(bashpid, "r");
            if (f)
            {
                fclose(f);
                remove(bashpid);
            }
        }
        if (pi.hProcess)
        {
            ::GenerateConsoleCtrlEvent(CTRL_BREAK_EVENT, pi.dwProcessId);
            Sleep(100);
            writeToPipe("\015\012");
            Sleep(100);
            ::TerminateProcess(pi.hProcess, 0);
            ::memset(&pi, 0, sizeof(PROCESS_INFORMATION));
            ::memset(&startInfo, 0, sizeof(STARTUPINFO));
			exitCode = 0;
        }
    }
    void write(const char *data)
    {
        writeToPipe(data);
    }
private:
    wstring appDirPath;
	char bashpid[100]; // Cygwin
	char* buffer;
	string bA;
	HANDLE hStdinRd, hStdinWr, hStdoutRd, hStdoutWr;
	PROCESS_INFORMATION pi;
	STARTUPINFO startInfo;
	DWORD exitCode;

    int readFromPipe(char *data, int len)
    {
        DWORD count;
        ::PeekNamedPipe(hStdoutRd, data, len, NULL, &count, NULL);
        if (count == 0)
            return 0;
        if (!::ReadFile(hStdoutRd, data, len, &count, NULL) || count == 0)
            return 0;
        return (int)count;
    }
    int writeToPipe(const char *data)
    {
        DWORD count;
        if (!::WriteFile(hStdinWr, data, (DWORD)strlen(data), &count, NULL))
            return 0;
        return (int)count;
    }
};
