/*
 Program WinCaml: Graphical User Interface
 for interactive use of Caml-Light and Ocaml.
 Copyright (C) 2005-2018 Jean Mouric 35700 Rennes France
 email: jean.mouric@orange.fr
 
 This program is free software: you can redistribute it and/or modify
 it under the terms of the GNU General Public License as published by
 the Free Software Foundation, either version 3 of the License, or
 (at your option) any later version.
 
 This program is distributed in the hope that it will be useful,
 but WITHOUT ANY WARRANTY; without even the implied warranty of
 MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 GNU General Public License for more details.
 
 You should have received a copy of the GNU General Public License
 along with this program. If not, see <http://www.gnu.org/licenses/>.
 */

// File Win/Paths.cpp  (included from Qt/Path.cpp and wx/Paths.cpp)

#ifdef _WINDOWS_PLATFORM_
#include "Win.h"
#endif
#include <fstream>
#include <sstream>
#include <shlobj.h>
#include "Shlwapi.h"
#include <string>
#include <algorithm>

extern bool camlnew;
bool dismissed = false;

enum{OCAML64_LOCAL, OCAML64, OCAML32_64, OCAML32_64A, OCAML32, OCAML32A, OCAML32B, CAMLLIGHT64, CAMLLIGHT64A, CAMLLIGHT32, CAMLLIGHT32A, CAMLLIGHT, OCAMLNOTFOUND, CAMLNOTFOUND};

#if defined (Q_OS_WIN) || defined (__WXMSW__)
bool getConfigFolderPath(wstring& path)
{
#ifdef Q_OS_WIN
	wchar_t szPath[MAX_PATH];
#else
	WCHAR szPath[MAX_PATH];
#endif
	if (::SHGetFolderPath(NULL, CSIDL_LOCAL_APPDATA, NULL, 0, szPath) == S_OK)
	{
		path = szPath;
		path += L"\\" + appName + L"\\";
		return true;
	}
	return false;
}

wstring applicationDirectoryPath()
{
#ifdef Q_OS_WIN
	wchar_t appPath[MAX_PATH];
#else
	WCHAR appPath[MAX_PATH];
#endif
	::GetModuleFileName(0, appPath, sizeof(appPath) - 1);
	wstring appDirPath = appPath;
	return appDirPath.substr(0, appDirPath.find_last_of(L"\\"));
}

#ifdef QT4_7
typedef BOOL(WINAPI *LPFN_ISWOW64PROCESS) (HANDLE, BOOL*);
LPFN_ISWOW64PROCESS fnIsWow64Process;

static bool IsWow64()
{
	BOOL bIsWow64 = FALSE;
	//IsWow64Process is not available on all supported versions of Windows.
	//Use GetModuleHandle to get a handle to the DLL that contains the function
	//and GetProcAddress to get a pointer to the function if available.
	fnIsWow64Process = (LPFN_ISWOW64PROCESS) ::GetProcAddress(::GetModuleHandle(TEXT("kernel32")), "IsWow64Process");
	if (NULL != fnIsWow64Process)
	{
		if (!fnIsWow64Process(::GetCurrentProcess(), &bIsWow64))
		{
			//handle error
		}
	}
	return bIsWow64 == TRUE;
}

static bool architecture64()
{
	return IsWow64();
}
#else
static bool architecture64()
{
	SYSTEM_INFO sysInfo;
	::GetNativeSystemInfo(&sysInfo);
	return sysInfo.wProcessorArchitecture == PROCESSOR_ARCHITECTURE_AMD64;
}
#endif
#endif

extern CMDIFrame* mainFrame;

void ldconf(wstring& ocamlPath)
{
    wstring ocamlLibPath = ocamlPath + L"\\lib\\ocaml";
    if (!PathFileExists(ocamlLibPath.c_str())) { ocamlLibPath = ocamlPath + L"\\lib"; }
    
    wstring ldconfPath = ocamlLibPath +  L"\\ld.conf";

#if defined(_MINGW_) || defined(_MSVC98_)
    string toBytes(const wstring& wstr);
    fstream fs(toBytes(ldconfPath).c_str(), fstream::out);
#else
    fstream fs(ldconfPath.c_str(), fstream::out);
#endif

    if (!fs.fail())
    {
		string fromUnicode(const wstring& wstr, int enc);
		string s = fromUnicode(ocamlLibPath, ENCODING_UTF8);
        fs << (s + "\\stublibs").c_str() << endl;
        fs << s.c_str();
        fs.close();
    }
}

bool getConfigFolderPath(wstring& path);

wstring getCamlPath(bool ocaml)
{
    wstring cfp;
    string camlPath = "";
    if (getConfigFolderPath(cfp))
    {
        cfp += ocaml ? L"ocamlbase.txt" : L"camlbase.txt";
#if defined(_MINGW_) || defined(_MSVC98_)
        string toBytes(const wstring& wstr);
        fstream fs(toBytes(cfp).c_str(), fstream::in);
#else
        fstream fs(cfp.c_str(), fstream::in);
#endif
        if (!fs.fail())
        {
            char c;
            while (fs.get(c) && c != L'\r' && c != L'\n')
                camlPath += c;
            
            fs.close();
        }
    }
    wstring toUnicode(const string& str, int enc);
    return toUnicode(camlPath, ENCODING_UTF8);
}

void writeCamlPath(wstring camlPath, wstring camlBase)
{
    wstring s;
    if (getConfigFolderPath(s))
    {
        s += camlBase;
#if defined(_MINGW_) || defined(_MSVC98_)
        string toBytes(const wstring& wstr);
        fstream fs(toBytes(s).c_str(), fstream::out);
#else
        fstream fs(s.c_str(), fstream::out);
#endif
        if (!fs.fail())
        {
            string fromUnicode(const wstring& wstr, int enc);
            fs << fromUnicode(camlPath, ENCODING_UTF8) << endl;
            fs.close();
        }
    }
}

wstring setCamlPath(bool ocaml)
{
	dismissed = false;
    wstring distrib = ocaml ? L"OCaml" : L"Caml Light";
	wstring message = L"Le dialogue suivant devrait permettre de naviguer vers un dossier " + distrib + L" valide ou d'annuler.";
    ::infoMessage(message);
    WCHAR path[MAX_PATH];
	memset(path, 0, sizeof(path));
    BROWSEINFO bi = {0};
    bi.lpszTitle = (ocaml ? L"Emplacement d'OCaml ?" : L"Emplacement de Caml Light ?");
#ifdef _WINDOWS_PLATFORM_
	bi.hwndOwner = mainFrame->hWnd;
#endif
#ifdef __WXMSW__
	bi.hwndOwner = (HWND)mainFrame->GetHWND();
#endif
#ifdef Q_OS_WIN
	bi.hwndOwner = (HWND)mainFrame->winId();
#endif
    LPITEMIDLIST pidl = SHBrowseForFolder(&bi);
    if (pidl != 0)
    {
        SHGetPathFromIDList (pidl, path);
        IMalloc* p = NULL;
        if ( SUCCEEDED(SHGetMalloc(&p)))
        {
            p->Free (pidl);
            p->Release();
        }
        return path;
	}
	dismissed = true;
    return L"";
}

bool resetCamlPath(const wstring& camlbase)
{
    wstring s;
    if (getConfigFolderPath(s))
    {
        s += camlbase;
        if (PathFileExists(s.c_str()))
        {
			wstring distrib = camlbase == L"camlbase.txt" ? L"Caml Light" : L"OCaml";
			wstring message = L"Abandonner la distribution " + distrib + L" actuelle ?";
			if (::yesnoMessage(message))
			{
				void deleteFile(const wstring& fileName);
				deleteFile(s);
			}
			return true;
        }
    }
	return false;
}

bool resetCamlPaths()
{
    bool b1 = resetCamlPath(L"camlbase.txt");
    bool b2 = resetCamlPath(L"ocamlbase.txt");
	return b1 || b2;
}

bool architecture64();

bool isSixtyFour(const wstring& path)
{
    unsigned char t[2];
	FILE* f = _wfopen((wchar_t*)path.c_str(), L"rb");
	if (f)
    {
        int i = 0;
        while (fread(&t, 2, 1, f) == 1 && (t[0] != 0x50 || t[1] != 0x45))
        {
            i++;
        }
        if (fread(&t, 2, 1, f) == 1 && fread(&t, 2, 1, f) == 1)
        {
            bool b = t[0] + t[1] * 0x100 == 0x8664;
			return b;
		}
    }
    return architecture64();
}

wstring ocamlCommandLine(wstring ocamlPath, wstring& dirPath, string& cygwin)
{
	if (ocamlPath == L"")
	{
		return L"";
	}
	wstring cmdl;
	string toBytes(const wstring& wstr);
	bool sixtyfour = isSixtyFour(ocamlPath + L"\\bin\\ocamlrun.exe");

	wstring libgraphDirPath = dirPath + L"\\ocaml\\windows";
	sixtyfour ? libgraphDirPath += L"\\ocaml64" : libgraphDirPath += L"\\ocaml32";

	wstring ocamlLibPath = ocamlPath + L"\\lib\\ocaml";
	if (PathFileExists(ocamlLibPath.c_str())) // Cygwin
	{
		wstring fromBytes(const string& str);
		string s1 = toBytes(ocamlPath);
		std::replace(s1.begin(), s1.end(), '\\', '/');
		char c = 'c';
		if (s1.length() > 2 && s1[1] == char(':'))
		{
			c = s1[0];
			s1 = string("/cygdrive/").append(s1.erase(1, 1));
		}
        string s2 = toBytes(libgraphDirPath);
        std::replace(s2.begin(), s2.end(), '\\', '/');
        if (s2.length() > 2 && s2[1] == char(':'))
        {
            s2 = string("/cygdrive/").append(s2.erase(1, 1));
        }
		cygwin = string(1, c) + ":\\cygwin";
		if (sixtyfour) cygwin += "64";
		wstring cygwinW = fromBytes(cygwin);
		cmdl = cygwinW + L"\\bin\\bash.exe -c 'ocamlrun ocaml -I \"$LIBGRAPHDIRPATH\" -I \"$OCAMLPATH/lib/ocaml/stublibs\"'";
		::SetEnvironmentVariable(wstring(L"OCAMLLIB").c_str(), fromBytes(s1 + "/lib/ocaml").c_str());
		::SetEnvironmentVariable(wstring(L"PATH").c_str(), (ocamlPath + L"\\bin;" + cygwinW + L"\\bin;C:\\Windows\\System32;C:\\Windows").c_str());
		::SetEnvironmentVariable(wstring(L"CAML_LD_LIBRARY_PATH").c_str(), fromBytes(s1 + "/lib/ocaml/stublibs").c_str());
		::SetEnvironmentVariable(wstring(L"OCAMLPATH").c_str(), fromBytes(s1).c_str());
        ::SetEnvironmentVariable(wstring(L"LIBGRAPHDIRPATH").c_str(), fromBytes(s2).c_str());
		
	}
	else
	{
		wstring ocamlLibPath = ocamlPath + L"\\lib";
		wstring ocamlStublibsPath = ocamlPath + L"\\lib\\stublibs";
		wstring ocamlrun = L"\"" + ocamlPath + L"\\bin\\ocamlrun.exe\"" + L" \"" + ocamlPath + L"\\bin\\ocaml.exe\"";
		cmdl = ocamlrun + L" -I \"" + libgraphDirPath + L"\" -I \"" + ocamlLibPath + L"\" -I \"" + ocamlStublibsPath + L"\"";
		::SetEnvironmentVariable(wstring(L"PATH").c_str(), (ocamlPath + L"\\bin;C:\\Windows\\System32;C:\\Windows").c_str());
		::SetEnvironmentVariable(wstring(L"OCAMLLIB").c_str(), ocamlLibPath.c_str());
		::SetEnvironmentVariable(wstring(L"CAML_LD_LIBRARY_PATH").c_str(), ocamlStublibsPath.c_str());
		ldconf(ocamlPath);
	}
	dirPath = ocamlPath;
	return cmdl;
}

wstring camllightCommandLine1(const wstring& dirPath, string& cygwin, string arch)
{
    if (dirPath == L"")
    {
        return L"";
    }
	wstring fromBytes(const string& str);
	string toBytes(const wstring& wstr);
	wstring tpl = dirPath + L"\\caml-light\\bin\\camltoplevel" + fromBytes(arch) + L"cygwin.exe";
	if (!PathFileExists(tpl.c_str())) {
		return L"unknown";
	}
    string dirPath1 = toBytes(dirPath);
    std::replace(dirPath1.begin(), dirPath1.end(), '\\', '/');
    char c = 'c';
    if (dirPath1.length() > 2 && dirPath1[1] == char(':'))
    {
        c = dirPath1[0];
        dirPath1 = string("/cygdrive/").append(dirPath1.erase(1, 1));
    }

    string archi = arch == "64" ? "64" : "";
    cygwin = string(1, c) + ":\\cygwin" + archi;
    wstring cygwinW = fromBytes(cygwin);
    
    wstring cmdl = cygwinW + L"\\bin\\bash.exe -c '\"" + fromBytes(dirPath1) + L"/caml-light/bin/camltoplevel\"" + fromBytes(arch) + L"cygwin -stdlib \"" + fromBytes(dirPath1) + L"\"/caml-light/lib/caml-light -lang \"fr\"'";
    ::SetEnvironmentVariable(wstring(L"PATH").c_str(), (dirPath + L"\\caml-light\\bin;" + cygwinW + L"\\bin;C:\\Windows\\System32;C:\\Windows").c_str());
    ::SetEnvironmentVariable(wstring(L"CAMLLIGHTDIR").c_str(), fromBytes(dirPath1 + "/caml-light").c_str());
	cygwin = string(1, c) + ":\\cygwin" + archi;
	return cmdl;
}

int isCygwin(const wstring& path)
{
    char* q = (char*)"cygwin1.dll";
    char* p = q;
    char c;
    FILE* f = _wfopen(path.c_str(), L"rb");
    if (!f) return 0;
    while (fread(&c, 1, 1, f) == 1 && *p != 0)
    {
        if (*p != c) p = q ; else p++;
    }
    fclose(f);
    return *p == 0;
    
}

wstring camllightCommandLine2(const wstring& camlPath, string& cygwin)
{
    if (camlPath == L"")
    {
        return L"";
    }
    wstring fromBytes(const string& str);
    string toBytes(const wstring& wstr);
    
    wstring tpl = camlPath + L"\\lib\\caml-light\\caml_all.exe";
    if (!PathFileExists(tpl.c_str()))
    {
        return L"unknown";
    }
    
    wstring cmdl;
	if (isCygwin(tpl))
    {
        string camlPath1 = toBytes(camlPath);
        std::replace(camlPath1.begin(), camlPath1.end(), '\\', '/');
        char c = 'c';
        if (camlPath1.length() > 2 && camlPath1[1] == char(':'))
        {
            c = camlPath1[0];
            camlPath1 = string("/cygdrive/").append(camlPath1.erase(1, 1));
        }

        wstring caml_allPath = camlPath + L"\\lib\\caml-light\\caml_all.exe";
        string archi = isSixtyFour(caml_allPath) ? "64" : "";
        cygwin = string(1, c) + ":\\cygwin" + archi;
        wstring cygwinW = fromBytes(cygwin);
        
        cmdl = cygwinW + L"\\bin\\bash -c '\"$CAMLLIGHTDIR/lib/caml-light/caml_all\" -stdlib \"$CAMLLIGHTDIR/lib/caml-light\" -lang \"fr\"'";
        ::SetEnvironmentVariable(wstring(L"PATH").c_str(), (camlPath + L"\\caml-light\\bin;" + cygwinW + L"\\bin;C:\\Windows\\System32;C:\\Windows").c_str());
        ::SetEnvironmentVariable(wstring(L"CAMLLIGHTDIR").c_str(), fromBytes(camlPath1).c_str());
    }
    else
    {
        ::SetEnvironmentVariable(wstring(L"PATH").c_str(), (camlPath + L"\\bin;C:\\Windows\\System32;C:\\Windows").c_str());
        ::SetEnvironmentVariable(wstring(L"CAMLLIGHTDIR").c_str(), camlPath.c_str());
        cmdl = L"\"" + camlPath + L"\\lib\\caml-light\\caml_all.exe\"" + L" -stdlib \"" + camlPath + L"\\lib\\caml-light\"" + L" -lang \"fr\"";
    }
    return cmdl;
}

wstring commandLine(int option, wstring& dirPath, string& cygwin)
{
    wstring cmdl;
    cygwin = "";
    switch(option)
    {
        case OCAML64_LOCAL:
        {
			cmdl = ocamlCommandLine(dirPath + L"\\ocaml\\windows\\ocaml64\\miniocaml", dirPath, cygwin);
            break;
        }
        case OCAML64:
        {
			cmdl = ocamlCommandLine(L"C:\\Program Files\\Objective Caml", dirPath, cygwin);
			break;
        }
        case OCAML32_64:
        {
			cmdl = ocamlCommandLine(L"C:\\Program Files (x86)\\Objective Caml", dirPath, cygwin);
			break;
        }
        case OCAML32_64A:
        {
			cmdl = ocamlCommandLine(L"C:\\OCaml", dirPath, cygwin);
			break;
        }
        case OCAML32:
        {
			cmdl = ocamlCommandLine(L"C:\\Program Files\\Objective Caml", dirPath, cygwin);
			break;
        }
        case OCAML32A:
        {
			cmdl = ocamlCommandLine(L"C:\\OCaml", dirPath, cygwin);
			break;
        }
        case OCAML32B:
        {
			cmdl = ocamlCommandLine(getCamlPath(true), dirPath, cygwin);
			break;
		}
		case OCAMLNOTFOUND:
		{
            wstring camlPath = setCamlPath(true);
            cmdl = ocamlCommandLine(camlPath, dirPath, cygwin);
			break;
		}
        case CAMLLIGHT64:
        {
			cmdl = camllightCommandLine1(dirPath, cygwin, "64");
			break;
        }
        case CAMLLIGHT64A:
        {
            wstring locale = L"fr";
            cmdl = L"\"" + dirPath + L"\\caml-light\\bin\\camltoplevel64.exe\"" + L" -stdlib \"" + dirPath + L"\\caml-light\\lib\\caml-light\"" + L" -lang \"" + locale + L"\"";
            ::SetEnvironmentVariable(wstring(L"PATH").c_str(), (dirPath + L"\\caml-light\\bin;C:\\Windows\\System32;C:\\Windows").c_str());
            ::SetEnvironmentVariable(wstring(L"CAMLLIGHTDIR").c_str(), (dirPath + L"\\caml-light").c_str());
            break;
        }
        case CAMLLIGHT32:
        {
			cmdl = camllightCommandLine1(dirPath, cygwin, "32");
            break;
        }
        case CAMLLIGHT32A:
        {
		    wstring locale = L"fr";
            cmdl = L"\"" + dirPath + L"\\caml-light\\bin\\camltoplevel32.exe\"" + L" -stdlib \"" + dirPath + L"\\caml-light\\lib\\caml-light\"" + L" -lang \"" + locale + L"\"";
            ::SetEnvironmentVariable(wstring(L"PATH").c_str(), (dirPath + L"\\caml-light\\bin;C:\\Windows\\System32;C:\\Windows").c_str());
            ::SetEnvironmentVariable(wstring(L"CAMLLIGHTDIR").c_str(), (dirPath + L"\\caml-light").c_str());
           break;
        }
		case CAMLLIGHT:
		{
			cmdl = camllightCommandLine2(getCamlPath(false), cygwin);
			break;
		}
		case CAMLNOTFOUND:
		{
            dirPath = setCamlPath(false);
            cmdl = camllightCommandLine2(dirPath, cygwin);
			break;
		}
		default:
            cmdl = L"";
            break;
    }
    return cmdl;
}

wstring applicationDirectoryPath();

CProcess* startCaml1(bool ocaml)
{
	bool arch64 = architecture64();
	int option;
	if (ocaml && arch64) { option = OCAML64_LOCAL; }
	else if (ocaml && !arch64) { option = OCAML32A; }
	else if (!ocaml && arch64) { option = CAMLLIGHT64; }
	else { option = CAMLLIGHT32; }

	wstring dirPath = applicationDirectoryPath();

	CProcess* camlProcess = new CProcess();


start:
	string cygwin = "";
	wstring cmdl = commandLine(option, dirPath, cygwin);
	if (cmdl == L"")
	{
		if (ocaml && option != OCAMLNOTFOUND)
		{
			option = OCAMLNOTFOUND;
			goto start;
		}
		else if (!ocaml && option != CAMLNOTFOUND)
		{
			option = CAMLNOTFOUND;
			goto start;
		}
		else
		{
			delete camlProcess;
			return NULL;
		}
	}
	if (camlProcess->start(cmdl))
	{
		if (option == OCAMLNOTFOUND) writeCamlPath(dirPath, L"ocamlbase.txt");
		if (option == CAMLNOTFOUND) writeCamlPath(dirPath, L"camlbase.txt");
		if (cygwin != "") camlProcess->setPid(cygwin);
		return camlProcess;
	}
	switch (option)
	{
	case OCAML64_LOCAL:
	{
		option = OCAML64;
		goto start;
	}
	case OCAML64:
	{
		option = OCAML32_64A;
		goto start;
	}
	case OCAML32_64A:
	{
		option = OCAML32_64;
		goto start;
	}
	case OCAML32_64:
	{
		option = OCAML32B;
		goto start;
	}
	case OCAML32A:
	{
		option = OCAML32;
		goto start;
	}
	case OCAML32:
	{
		option = OCAML32B;
		goto start;
	}
	case OCAML32B:
	{
		option = OCAMLNOTFOUND;
		goto start;
	}
	case CAMLLIGHT64:
	{
		option = CAMLLIGHT64A;
		goto start;
	}
	case CAMLLIGHT64A:
	{
		option = CAMLLIGHT32;
		goto start;
	}
	case CAMLLIGHT32:
	{
		option = CAMLLIGHT32A;
		goto start;
	}
	case CAMLLIGHT32A:
	{
		option = CAMLLIGHT;
		goto start;
	}
	case CAMLLIGHT:
	{
		option = CAMLNOTFOUND;
		goto start;
	}
	case OCAMLNOTFOUND:
	{
		option = OCAMLNOTFOUND;
		goto start;
	}
	case CAMLNOTFOUND:
	{
		option = CAMLNOTFOUND;
		goto start;
	}
	default:
	{
		delete camlProcess;
		return NULL;
	}
	}
}


CProcess* startCaml(bool ocaml)
{
	if (camlnew)
	{
		camlnew = false;
		wstring message = L"Changer de distribution ";
		message += (ocaml ? L"OCaml ?" : L"Caml Light ?");
		if (::yesnoMessage(message))
		{
			string cygwin = "";
			wstring cmdl = L"";
			CProcess* camlProcess = new CProcess();
			wstring dirPath = applicationDirectoryPath();
			dismissed = false;
			while (!dismissed)
			{
				wstring path = setCamlPath(ocaml);
				cmdl = ocaml ? ocamlCommandLine(path, dirPath, cygwin) : camllightCommandLine2(path, cygwin);
				if (cmdl != L"" && camlProcess->start(cmdl))
				{
					writeCamlPath(path, ocaml ? L"ocamlbase.txt" : L"camlbase.txt");
					camlnew = false;
					if (cygwin != "") camlProcess->setPid(cygwin);
					return camlProcess;
				}
			}
			delete camlProcess;
			camlProcess = NULL;
		}
		if (dismissed && getCamlPath(ocaml) != L"")
		{
			if (!yesnoMessage(L"Garder la configuration actuelle ?"))
			{
				wstring cfp;
				if (getConfigFolderPath(cfp))
				{
					cfp += ocaml ? L"ocamlbase.txt" : L"camlbase.txt";
					void deleteFile(const wstring& fileName);
					deleteFile(cfp);
				}
			}
		}
		dismissed = false;
	}
	wstring path = getCamlPath(ocaml);
	if (path != L"")
	{
		string cygwin = "";
		wstring dirPath = applicationDirectoryPath();
		wstring cmdl = ocaml ? ocamlCommandLine(path, dirPath, cygwin) : camllightCommandLine2(path, cygwin);
		if (cmdl != L"")
		{
			CProcess* camlProcess = new CProcess();
			if (camlProcess->start(cmdl))
			{
				if (cygwin != "") camlProcess->setPid(cygwin);
				return camlProcess;
			}
		}
	}
	return startCaml1(ocaml);
}

void camllightHelp()
{
	wstring appDirPath = applicationDirectoryPath();
	wstring docPath = appDirPath + L"\\doc\\man-caml\\index.html";
	::ShellExecute(NULL, L"open", docPath.c_str(), NULL, appDirPath.c_str(), SW_SHOW);
}

void ocamlHelp()
{
	wstring appDirPath = applicationDirectoryPath();
	wstring docPath = appDirPath;
	docPath += L"\\doc\\htmlman\\index.html";
	if (32 >= (size_t)::ShellExecute(NULL, L"open", docPath.c_str(), NULL, appDirPath.c_str(), SW_SHOW))
	{
		if (32 >= (size_t)::ShellExecute(NULL, L"open", L"C:\\Program Files\\Objective Caml\\htmlman\\index.html", NULL, L"C:\\Program Files\\Objective Caml\\htmlman", SW_SHOW))
			::ShellExecute(NULL, L"open", L"C:\\Program Files (x86)\\Objective Caml\\htmlman\\index.html", NULL, L"C:\\Program Files (x86)\\Objective Caml\\htmlman", SW_SHOW);
	}
}
